/*
 *	RollerCoaster2000
 *	Copyright (C) 2003 Plusplus (plusplus@free.fr)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#import "RollerCoaster_2000View.h"
#import <gl.h>
#import <sys/time.h>

static	NSString*	trackNames[] = 
{
	@"rc2k",
	@"freestyle",
	@"hangtime2",
	@"quadraphase",
	@"midnight madness",
	@"devils tower",
	@"speedy loops",
	@"dizzy butterfly",
	@"wildemaus",
	@"wildemaus 2"
};

@implementation RollerCoaster_2000View

/**
*	initializes the screensaver object
*	@param	frame	the view rect
*	@param	preview	is it the preview ?
*	@return self
*/

- (id)initWithFrame:(NSRect)frame isPreview:(BOOL)preview
{
	NSOpenGLPixelFormat				*format;
	NSOpenGLPixelFormatAttribute	attrs[] =
	{
		NSOpenGLPFAAccelerated,
		NSOpenGLPFADoubleBuffer,
		NSOpenGLPFADepthSize, (NSOpenGLPixelFormatAttribute)32,
		(NSOpenGLPixelFormatAttribute)0
	};
	
	self = [super initWithFrame:frame isPreview:preview];
	if (self)
	{
		_delta = 0;
		_lastTime = 0;
		_view = NULL;
		_roller = NULL;
		_isModified = FALSE;
		_width = (int) frame.size.width;
		_height = (int) frame.size.height;
		
		_defaults = [[ScreenSaverDefaults defaultsForModuleWithName:[[NSBundle bundleForClass: [self class]] bundleIdentifier]] retain];
		if ([_defaults stringForKey:@"track"]==NULL) [self setDefaults];

		if (preview) _isMainScreen = TRUE;
		else
		{
			_isMainScreen = (frame.origin.x==0 && frame.origin.y==0) ? TRUE : FALSE;
			if (!_isMainScreen && ![_defaults boolForKey:@"mainScreenOnly"]) _isMainScreen = TRUE;
		}

		if (_isMainScreen)
		{
			format = [[[NSOpenGLPixelFormat alloc] initWithAttributes:attrs] autorelease];
			if (format != NULL)
			{
				_view = [[[NSOpenGLView alloc] initWithFrame:NSZeroRect pixelFormat:format] autorelease];
				if (_view == NULL) _isMainScreen = FALSE;
				else
				{
					[self addSubview:_view];
					[_view setFrameSize:frame.size];
					[self setAnimationTimeInterval:1.0/1000.0];
				}
			}
		}
	}

	return self;
}

/**
*	frees the screensaver object
*/

- (void)dealloc
{
	if (_view != NULL) [[_view openGLContext]makeCurrentContext];
	if (_roller != NULL)
	{
		delete _roller;
		_roller = NULL;
	}

	if (_view != NULL) [_view removeFromSuperview];
	[_defaults synchronize];
	[_defaults release];
	[super dealloc];
}

/**
*	fills the screen with black
*/

- (void) drawRect:(NSRect) inFrame
{
	[[NSColor blackColor] set];            
    NSRectFill(inFrame);
}

/**
*	starts the screensaver animation
*/

- (void)startAnimation
{
	BOOL		showFrames;
	int			random;
	NSBundle	*bundle;
	NSString	*path,*track,*name;

	if (_isMainScreen)
	{
		[[_view openGLContext]makeCurrentContext];
	
		if ([self isPreview]) showFrames = FALSE;
		else
		{
			showFrames = [_defaults boolForKey:@"showFPS"];
			glClearColor(0.0f,0.0f,0.0f,1.0f);
			glClear(GL_COLOR_BUFFER_BIT);
			[[_view openGLContext]flushBuffer];
		}
	
		if (_roller == NULL || _isModified)
		{
			_isModified = FALSE;
			_lastTime = 0;
			_delta = 0;
			
			if (_roller != NULL)
			{
				delete _roller;
				_roller = NULL;
			}
			
			name = [_defaults stringForKey:@"track"];
			if ([name isEqualToString:@"Random"])
			{
				random = SSRandomIntBetween(0,(sizeof(trackNames)/sizeof(NSString*))-1);
				name = trackNames[random];
			}
			
			bundle = [NSBundle bundleForClass: [self class]];
			path = [bundle resourcePath];
			track = [bundle pathForResource:name ofType:@"trk"];
		
			_roller = new RollerCoaster();
			_roller->Start((char*)[path cString],(char*)[track cString],_width,_height,[_defaults integerForKey:@"intro"],[_defaults integerForKey:@"details"],showFrames);
		}
	
		if (_lastTime == 0) _lastTime = [self getTime];
		_delta += ([self getTime] - _lastTime);	
	}

	[super startAnimation];
}

/**
*	stops the screensaver animation
*/

- (void)stopAnimation
{
	[super stopAnimation];
	if (_isMainScreen) _lastTime = [self getTime];
}

- (void)animateOneFrame
{
	unsigned long	current;

	if (_isMainScreen)
	{
		if (_roller != NULL)
		{
			[[_view openGLContext]makeCurrentContext];	
			current = [self getTime];
			_roller->DrawRoller(current-_delta);
			[[_view openGLContext]flushBuffer];
		}
	}
}

- (BOOL)hasConfigureSheet
{
	return YES;
}

- (void)setDefaults
{
	[_defaults setObject:@"rc2k" forKey:@"track"];
	[_defaults setBool:TRUE forKey:@"intro"];
	[_defaults setInteger:50 forKey:@"details"];
	[_defaults setBool:FALSE forKey:@"showFPS"];
	[_defaults setBool:TRUE forKey:@"mainScreenOnly"];
}

- (void)setSheetValues
{
	[tracks selectItemWithTitle:[_defaults stringForKey:@"track"]];
	[flyby setState:[_defaults boolForKey:@"intro"]];
	[details setIntValue:[_defaults boolForKey:@"details"]];
	[showFPS setState:[_defaults boolForKey:@"showFPS"]];
	[mainScreenOnly setState:[_defaults boolForKey:@"mainScreenOnly"]];
}

- (IBAction)do_OK:(id)sender
{
	_isModified = TRUE;
	[_defaults setObject:[tracks titleOfSelectedItem] forKey:@"track"];
	[_defaults setBool:[flyby state] forKey:@"intro"];
	[_defaults setInteger:[details intValue] forKey:@"details"];
	[_defaults setBool:[showFPS state] forKey:@"showFPS"];
	[_defaults setBool:[mainScreenOnly state] forKey:@"mainScreenOnly"];
    [NSApp endSheet:panel];
}

- (IBAction)do_Cancel:(id)sender
{
    [NSApp endSheet:panel];
}

- (IBAction)do_Reset:(id)sender
{
	_isModified = TRUE;
	[self setDefaults];
	[self setSheetValues];
}

- (NSWindow*)configureSheet
{
	if (panel == nil)
    {
        [NSBundle loadNibNamed:@"Preferences" owner:self];
		for (int i=(sizeof(trackNames)/sizeof(NSString*))-1;i>=0;i--) [tracks insertItemWithTitle:trackNames[i] atIndex:0];
		[self setSheetValues];
    }
    
    return panel;
}

/**
*	returns the time in milliseconds
*/

- (unsigned long)getTime
{
	struct timeval tv;

	gettimeofday(&tv,NULL);
	return tv.tv_sec*1000 + tv.tv_usec/1000;
}

@end
