/*
 *	RollerCoaster2000
 *	Copyright (C) 2003 Plusplus (plusplus@free.fr)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "RollerCoaster.h"

void	RollerCoaster::mult(point *r, float t, point *p)
{
	r->x = t * p->x;
	r->y = t * p->y;
	r->z = t * p->z;
}

void	RollerCoaster::add_mult(point *r, float t, point *p)
{
	r->x += t * p->x;
	r->y += t * p->y;
	r->z += t * p->z;
}

float 	RollerCoaster::norme2(point *p)
{
	return p->x*p->x + p->y*p->y + p->z*p->z;
}

float 	RollerCoaster::norme(point *p)
{
	return (float)sqrt(p->x*p->x + p->y*p->y + p->z*p->z);
}

void 	RollerCoaster::normalize(point *p)
{
	register float n = norme(p);
	if(n == 0.0f) return;
	n = 1. / n;
	p->x = p->x * n;
	p->y = p->y * n;
	p->z = p->z * n;
}

void 	RollerCoaster::sub(point *r, point *a, point *b)
{
	r->x = a->x - b->x;
	r->y = a->y - b->y;
	r->z = a->z - b->z;
}

void 	RollerCoaster::add(point *r, point *a, point *b)
{
	r->x = a->x + b->x;
	r->y = a->y + b->y;
	r->z = a->z + b->z;
}

float 	RollerCoaster::prod_scal(point *a, point *b)
{
	return a->x * b->x + a->y * b->y + a->z * b->z;
}

void 	RollerCoaster::prod_vect(point *r, point *a, point *b)
{
	r->x = a->y * b->z - a->z * b->y;
	r->y = a->z * b->x - a->x * b->z;
	r->z = a->x * b->y - a->y * b->x;
}

// faire tourner le point p de l'angle t par rapport a l'axe de direction v
// le vecteur direction v doit etre unitaire

void 	RollerCoaster::rotate(point *r, point *v, float t, point *p)
{
	register float	cost,sint,mcost;
	register float	px,py,pz,vx,vy,vz;
	
	cost = cos(t);
	sint = sin(t);
	mcost = 1.0f - cost;
	px = p->x;
	py = p->y;
	pz = p->z;
	vx = v->x;
	vy = v->y;
	vz = v->z;
	
	r->x = ( vx * vx + cost * (1.0f - (vx * vx)) ) * px +
	       ( vx * vy * mcost - vz * sint ) * py +
	       ( vz * vx * mcost + vy * sint ) * pz;
	r->y = ( vx * vy * mcost + vz * sint ) * px +
	       ( vy * vy + cost * (1.0f - (vy * vy)) ) * py +
	       ( vy * vz * mcost - vx * sint ) * pz;
	r->z = ( vz * vx * mcost - vy * sint ) * px +
	       ( vy * vz * mcost + vx * sint ) * py +
	       ( vz * vz + cost * (1.0f - (vz * vz)) ) * pz;
}

// calcul le point d'intersection de la droite passant par a de direction v avec le plan passant par p de normale n
// renvoie 0 si la droite est parallle au plan

int 	RollerCoaster::intersection(point *r, point *a, point *v, point *p, point *n)
{
	point b;
	float t;
	sub(&b,p,a);
	t = prod_scal(v,n);
	if(t == 0.0f) return 0;
	t = prod_scal(&b,n) / t;
	r->x = a->x + t * v->x;
	r->y = a->y + t * v->y;
	r->z = a->z + t * v->z;
	return 1;
}

void	RollerCoaster::matrix_mult(float *r, float* a, float *b)
{
	int i,j;
	for(i=0 ; i<4 ; i++){
		for(j=0 ; j<4 ; j++){
			r[i+4*j] = a[i]*b[4*j] + a[i+4]*b[4*j+1] + a[i+8]*b[4*j+2] + a[i+12]*b[4*j+3];
		}
	}
}

void	RollerCoaster::mult_vect(point *r, float *m, point *v)
{
	register float w,vx,vy,vz;
	vx = v->x;
	vy = v->y;
	vz = v->z;
	w = 1.0f / (m[3]*vx + m[7]*vy + m[11]*vz + m[15]);
	r->x = (m[0]*vx + m[4]*vy + m[8]*vz + m[12]) * w;
	r->y = (m[1]*vx + m[5]*vy + m[9]*vz + m[13]) * w;
	r->z = m[2]*vx + m[6]*vy + m[10]*vz + m[14];
}
