/*
 *	RollerCoaster2000
 *	Copyright (C) 2003 Plusplus (plusplus@free.fr)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "RollerCoaster.h"
#include "cave.h"
#include <glu.h>
#include "TextureString.h"

/* This Will Be Called Right After The GL Window Is Created */

void	RollerCoaster::InitGL()
{
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LESS);
	glClearColor(0.3f, 0.3f, 0.7f, 0.0f); /* This will clear the background color to "la couleur du ciel" */
	glClearDepth(1.0);                    /* Enables clearing of the depth buffer */
	glDisable(GL_LINE_SMOOTH);
	glDisable(GL_POINT_SMOOTH);
	glDisable(GL_POLYGON_SMOOTH);
	glShadeModel(GL_SMOOTH);              /* Enables Gouraud shading */
	glDisable(GL_LIGHTING);
	glDisable(GL_BLEND);
	glDisable(GL_COLOR_MATERIAL);
	glDisable(GL_NORMALIZE);
	glDisable(GL_SCISSOR_TEST);
	glDisable(GL_STENCIL_TEST);

	glDisable(GL_DITHER);
	glEnable(GL_CULL_FACE);
	glCullFace(GL_BACK);
	glPolygonMode(GL_FRONT,GL_FILL);
	glFrontFace(GL_CCW);

	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}

void	RollerCoaster::ReSizeGLScene(int Width, int Height)
{
	float ratio, wd2;

	wndWidth = Width;
	wndHeight = Height;
//	glViewport(0, 0, Width, Height);   /* Reset the current viewport and perspective transformation */

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	ratio = (float)Width / (float)Height;
	wd2 = near_clip * tan(0.5f * M_PI * aperture / 180.0f);
	
	glFrustum(-ratio*wd2,  /* left   */
              ratio*wd2,   /* right  */
              -wd2,        /* bottom */
              wd2,         /* top    */
              near_clip, far_clip);

	glMatrixMode(GL_MODELVIEW);
}

// calcul la longueur d'un patch de Bzier en le discrtisant avec 1000 segments

float	RollerCoaster::longueur_bezier(point *p1, point *p2, point *p3, point *p4)
{
	point a,b;
	float t,f;
	float d = 1.0f/1000.0f;
	a = *p1;
	f = 0.0f;
	for(t=d ; t<=1.0f ; t+=d){
		mult(&b, (1-t)*(1-t)*(1-t), p1);
		add_mult(&b, 3*t*(1-t)*(1-t), p2);
		add_mult(&b, 3*t*t*(1-t), p3);
		add_mult(&b, t*t*t, p4);
		sub(&a,&b,&a);
		f += norme(&a);
		a = b;
	}
	return f;
}

/* Calcul l'inverse du rayon de courbure de la courbe de Bzier i (projet sur le plan xy)  l'abscisse t */

float 	RollerCoaster::rayonDeCourbure(int i, float t)
{
	point v,g; /* vitesse et acclration */

	v.z = g.z = 0.0f;
	mult(&v, -3*t*t+6*t-3, &pcontrole[3*i]);
	add_mult(&v, 9*t*t-12*t+3, &pcontrole[3*i+1]);
	add_mult(&v, -9*t*t+6*t, &pcontrole[3*i+2]);
	add_mult(&v, 3*t*t, &pcontrole[3*i+3]);
	mult(&g, -6*t+6, &pcontrole[3*i]);
	add_mult(&g, 18*t-12, &pcontrole[3*i+1]);
	add_mult(&g, -18*t+6, &pcontrole[3*i+2]);
	add_mult(&g, 6*t, &pcontrole[3*i+3]);
	v.z = g.z = 0.0f;


	/* formula : 1/R = det(v,g)/|v|^3 */
	if(norme2(&v) < 0.5f)  /* cas la trajectoire est proche de la verticale */
	{
		return 0.0f;
	}
	return (v.x * g.y - v.y * g.x) / norme2(&v) / norme(&v);
	/* autre faon de calculer l'inverse du rayon de courbure:
	 * projeter le vecteur acclration sur un vecteur unitaire orthoganal
	 *  la trajectoire ( l'aide d'un produit scalaire). Diviser le rsultat
	 * par le carr de la longueur du vecteur vitesse.
	 * 1 / R = prod_scal(&n,&g) / norme2(&v); o n est orthogonale  v
	 */
}

/* longueur_seg est la longueur d'un segment de la courbe */

void	RollerCoaster::InitCurve(float longueur_seg)
{
	int i,j,k;
	point a,b,c;
	float t,d;
	float *pnbsegment;

	/* initialise pcontrole  partir de ppointcontrole */
	pcontrole = (point*) malloc((3*nbPointControle+1)*sizeof(point));
	for(i=0 ; i<nbPointControle ; i++) {
		pcontrole[3*i] = ppointcontrole[2*i];
		pcontrole[3*i+1] = ppointcontrole[2*i+1];
	}
	pcontrole[3*nbPointControle] = pcontrole[0];
	/* calculer les points intermdiaires */
	for(i=0 ; i<3*nbPointControle ; i+=3){
		b = pcontrole[i+1];
		if(i>0) j = i-3; else j = 3*nbPointControle-3;
		if(norme(&b)<=0.1f){
			sub(&b,&pcontrole[i+3],&pcontrole[j]);
		}
		normalize(&b);
		sub(&c,&pcontrole[i+3],&pcontrole[i]);
		if(prod_scal(&b,&c)<0.0f){ b.x = -b.x; b.y = -b.y; b.z = -b.z; }
		mult(&a,norme(&c)*M_PI/5.5f/1.41421f,&b);
		add(&pcontrole[i+1],&pcontrole[i],&a);
		b.x = -b.x; b.y = -b.y; b.z = -b.z;
		sub(&c,&pcontrole[i],&pcontrole[j]);
		mult(&a,norme(&c)*M_PI/5.5f/1.41421f,&b);
		if(i>0) k = i-1; else k = 3*nbPointControle-1;
		add(&pcontrole[k],&pcontrole[i],&a);
	}
	/* calculer le nombre de segments pour chaque patch de Bzier */
	pnbsegment = (float*)malloc(nbPointControle*sizeof(float));
	nbLine = 0;
	for(i=0 ; i<nbPointControle ; i++){
		j = i * 3;
		t = longueur_bezier(&pcontrole[j],&pcontrole[j+1],&pcontrole[j+2],&pcontrole[j+3]);
		pnbsegment[i] = t / longueur_seg;
		if(!pnbsegment[i]) pnbsegment[i] = 1;
		nbLine += (int)pnbsegment[i] + 1;
	}
	/* faire la discrtisation pour tout les patchs */
	pline = (point*)malloc(nbLine*sizeof(point));
/*	pcourbure = (float*)malloc(nbLine*sizeof(float)); */
	k = 0;
	for(i=0 ; i<nbPointControle ; i++){
		a = pcontrole[3*i];
		pline[k++] = a;
		d = 1.0f / (pnbsegment[i]+1.0f);
		for(j=1,t=d ; j<pnbsegment[i] ; j++,t+=d){
			mult(&b, (1-t)*(1-t)*(1-t), &pcontrole[3*i]);
			add_mult(&b, 3*t*(1-t)*(1-t), &pcontrole[3*i+1]);
			add_mult(&b, 3*t*t*(1-t), &pcontrole[3*i+2]);
			add_mult(&b, t*t*t, &pcontrole[3*i+3]);
			pline[k] = b;
			if(b.z > maxZ) maxZ = b.z;
			if(b.x * b.x + b.y * b.y > maxDist) maxDist = b.x * b.x + b.y * b.y;
/*			pcourbure[k] = - rayonDeCourbure(i,t); */
			k++;
		}
	}
	maxDist = sqrt(maxDist);
	free(pnbsegment);
}

/* intersection du cylindre d'axe v avec le plan passant par p de normal n */

void 	RollerCoaster::calcCercle(point *p, point *z, point *v, int nbCote, float rayon, float angle, point *n, point *pmesh)
{
	point r;
	int i;
	for(i=0 ; i<nbCote ; i++){
		rotate(&r, v, (angle+(float)i)*2.0f*M_PI/(float)nbCote, z);
		normalize(&r);
		mult(&r,rayon,&r);
		add(&r,p,&r);
		intersection(&r,&r,v,p,n);
		pmesh[i] = r;
	}
}

void 	RollerCoaster::calcTuyau(point *pline, int nbLine, int nbCote, float rayon, float angle, point *pmesh)
{
	point a,b,c;
	point v,p;
	int i;
	a = pline[nbLine-1];
	for(i=0 ; i<nbLine ; i++){
		b = pline[i];
		c = pline[(i+1)%nbLine];
		sub(&v,&b,&a);
		normalize(&v);
		sub(&c,&c,&b);
		normalize(&c);
		add(&p,&v,&c);
		c = ppos[i];
		calcCercle(&b, &c, &v, nbCote, rayon, angle+pcourbure[i], &p, &pmesh[i*nbCote]);
		a = b;
	}
}

void 	RollerCoaster::moyenne(float *tab, int nbelem, int nbmoy_pred, int nbmoy_succ)
{
	float *buf;
	int i,j,k;
	float tot,f,m;
	buf = (float*)malloc(nbelem*sizeof(float));
	for(i=0 ; i<nbelem ; i++) buf[i] = tab[i];
	for(i=0 ; i<nbelem ; i++){
		k = i;
		tot = 0.0f;
		m = 0.0f;
		for(j=0 ; j<nbmoy_pred ; j++){
			f = (float)j / nbmoy_pred;
			f = 1.0f - f*f;
			m += f;
			tot += f * buf[k];
			k = (k-1+nbelem) % nbelem;
		}
		k = (i+1) % nbelem;
		for(j=1 ; j<nbmoy_succ ; j++){
			f = (float)j / nbmoy_succ;
			f = 1.0f - f*f;
			m += f;
			tot += f * buf[k];
			k = (k+1) % nbelem;
		}
		tab[i] = tot / m;
	}
	free(buf);
}

void 	RollerCoaster::InitLines()
{
	int i,j,k;
	float t;
	point a,b,c;
	point p,v;
	/* initialisation de pcourbure et pnorme */
	pcourbure = (float*)malloc(nbLine*sizeof(float));
	pnorme = (float*)malloc(nbLine*sizeof(float));
	k=0;
	for(i=0 ; i<nbLine ; i++){
		sub(&p,&pline[i],&pline[(i-1+nbLine)%nbLine]);
		sub(&v,&pline[(i+1)%nbLine],&pline[i]);
		pnorme[i] = norme(&p);
		normalize(&p);
		normalize(&v);
		p.z = v.z = 0;
		prod_vect(&a,&v,&p);
		pcourbure[i] = twistFactor * asin(a.z);
/*		pcourbure[i] = pcourbure[i] / 1.8f; */
	}
	moyenne(pcourbure,nbLine,7,5);
	/* initialisation de ppos */
	ppos = (point*) malloc(nbLine*sizeof(point));
	c.x = 0.0f; c.y = 0.0f; c.z = 1.0f;
	/************** at point j, curve's upward vector must be +z ************/
	j = 0;
	for(i=0 ; i<nbLine ; i++){
		a = pline[(j+i-1+nbLine)%nbLine];
		b = pline[(j+i)%nbLine];
		sub(&p,&b,&a);
		normalize(&p);
		t = -prod_scal(&p,&c)/* / prod_scal(&p,&p)*/;
		add_mult(&c,t,&p);
		normalize(&c);
		prod_vect(&a,&p,&c);
		if(a.z != 0.0f){
			a.z = 0.0f;
			prod_vect(&c,&a,&p);
			normalize(&c);
		}
		ppos[(i+j)%nbLine] = c;
	}
	/* initialisation de ptraj et ptang */
	ptraj = (point*) malloc(nbLine*sizeof(point));
	ptang = (point*) malloc(nbLine*sizeof(point));
	a = pline[nbLine-1];
	for(i=0 ; i<nbLine ; i++){
		b = pline[i];
		c = pline[(i+1)%nbLine];
		sub(&p,&b,&a);
		normalize(&p);
		sub(&c,&c,&b);
		normalize(&c);
		add(&v,&p,&c);
		ptang[i] = v;
		normalize(&ptang[i]);
		add(&c,&ppos[i],&b);
		intersection(&a,&c,&p,&b,&v);
		sub(&a,&a,&b);
		normalize(&a);
		ptraj[i] = a;
		a = b;
	}
	/* initialisation de pcyl */
	pcyl = (point*) malloc(nbLine*4*sizeof(point));
	calcTuyau(pline,nbLine,4,rayonCyl,0.0f,pcyl);
	/* initialisation de pline1 et pline2 */
	pline1 = (point*) malloc(nbLine*sizeof(point));
	pline2 = (point*) malloc(nbLine*sizeof(point));
	a = pline[nbLine-1];
	for(i=0 ; i<nbLine ; i++){
		b = pline[i];
		c = pline[(i+1)%nbLine];
		sub(&p,&b,&a);
		normalize(&p);
		sub(&v,&c,&b);
		normalize(&v);
		t = pcourbure[i];
		add(&p,&v,&p);
		normalize(&p);
		v = ptraj[i];
		rotate(&c,&p,t+1.0f*M_PI/3.0f,&v);
		mult(&c,longueurLien,&c);
		add(&pline1[i],&c,&b);
		rotate(&c,&p,t-1.0f*M_PI/3.0f,&v);
		mult(&c,longueurLien,&c);
		add(&pline2[i],&c,&b);
		a = b;
	}
	/* initialisation de prail1 */
	prail1 = (point*) malloc(nbLine*nbCote*sizeof(point));
	calcTuyau(pline1,nbLine,nbCote,rayonCyl,0.0f,prail1);
	/* initialisation de prail2 */
	prail2 = (point*) malloc(nbLine*nbCote*sizeof(point));
	calcTuyau(pline2,nbLine,nbCote,rayonCyl,0.0f,prail2);
}

// f est la distance entre 2 liens conscutif souhaite

void 	RollerCoaster::InitLiens(float f)
{
	point a,a1,a2,p,p1,p2,b,v,n;
	float longueur,l;
	int i;
	int ia,ib;
	float distLien;
	float t,t1,t2;
	longueur = 0.0f;
	for(i=0 ; i<nbLine ; i++){
		longueur += pnorme[i];
	}
	nbLiens = (int)floor(longueur / f);
	pliens = (point*)malloc(nbLiens*12*sizeof(point));
	pnliens = (point*)malloc(nbLiens*3*sizeof(point));
	distLien = longueur / (float)nbLiens;
	l = 0.0f;
	ia = nbLine-1;
	ib = 0;
	sub(&p,&pline[ib],&pline[ia]);
	sub(&p1,&pline1[ib],&pline1[ia]);
	sub(&p2,&pline2[ib],&pline2[ia]);
	for(i=0 ; i<nbLiens ; i++){
		while(l > pnorme[ib]){
			l -= pnorme[ib];
			ia = (ia + 1) % nbLine;
			ib = (ib + 1) % nbLine;
			sub(&p,&pline[ib],&pline[ia]);
			sub(&p1,&pline1[ib],&pline1[ia]);
			sub(&p2,&pline2[ib],&pline2[ia]);
		}
		t = l / pnorme[ib];
		a = pline[ia];
		add_mult(&a,t,&p);
		pnliens[i*3] = a;
		a1 = pline1[ia];
		sub(&v,&a,&a1);
		t1 = prod_scal(&v,&p) / prod_scal(&p1,&p);
		add_mult(&a1,t1,&p1);
		pnliens[i*3+1] = a1;
		a2 = pline2[ia];
		sub(&v,&a,&a2);
		t2 = prod_scal(&v,&p) / prod_scal(&p2,&p);
		add_mult(&a2,t2,&p2);
		pnliens[i*3+2] = a2;
		sub(&n,&a1,&a2);
		normalize(&n);
		sub(&v,&a1,&a);
		normalize(&v);
		prod_vect(&b,&v,&p);
		normalize(&b);
		calcCercle(&a,&b,&v,4,rayonLien,0.0f,&n,&pliens[i*12]);       // lien avec le rail central
		calcCercle(&a1,&b,&v,4,rayonLien,0.0f,&n,&pliens[i*12+4]);
		sub(&v,&a,&a2);
		normalize(&v);
		prod_vect(&b,&v,&p);
		normalize(&b);
		calcCercle(&a2,&b,&v,4,rayonLien,0.0f,&n,&pliens[i*12+8]);
		l += distLien;
	}
}

void 	RollerCoaster::InitNormales()
{
	int i;
	prail1n = (point*)malloc(nbLine*nbCote*sizeof(point));
	for(i=0 ; i<nbLine*nbCote ; i++){
		sub(&prail1n[i],&prail1[i],&pline1[i/nbCote]);
		normalize(&prail1n[i]);
	}
	prail2n = (point*)malloc(nbLine*nbCote*sizeof(point));
	for(i=0 ; i<nbLine*nbCote ; i++){
		sub(&prail2n[i],&prail2[i],&pline2[i/nbCote]);
		normalize(&prail2n[i]);
	}
	pcyln = (point*)malloc(nbLine*4*sizeof(point));
	for(i=0 ; i<nbLine*4 ; i++){
		sub(&pcyln[i],&pcyl[i],&pline[i/4]);
		normalize(&pcyln[i]);
	}
	pliensn = (point*)malloc(nbLiens*12*sizeof(point));
	for(i=0 ; i<nbLiens*3 ; i++){
		sub(&pliensn[i*4],&pliens[i*4],&pnliens[i]);
		normalize(&pliensn[i*4]);
		sub(&pliensn[i*4+1],&pliens[i*4+1],&pnliens[i]);
		normalize(&pliensn[i*4+1]);
		sub(&pliensn[i*4+2],&pliens[i*4+2],&pnliens[i]);
		normalize(&pliensn[i*4+2]);
		sub(&pliensn[i*4+3],&pliens[i*4+3],&pnliens[i]);
		normalize(&pliensn[i*4+3]);
	}
}

void 	RollerCoaster::InitCouleurs(point *light_dir)
{
	int i;
	float f;
	static GLfloat LightAmbient[] = {0.4f,0.4f,0.4f,1.0f};
	static GLfloat LightDiffuse[] = {1.1f,0.8f,1.1f,1.0f};

	normalize(light_dir);
	for(i=0 ; i<nbLine*nbCote ; i++){
		f = prod_scal(light_dir,&prail1n[i]);
		if(f<=0.0f) f = 0.0f;
		prail1n[i].x = LightDiffuse[0] * f + LightAmbient[0];
		prail1n[i].y = LightDiffuse[1] * f + 0.0f * LightAmbient[1];
		prail1n[i].z = LightDiffuse[2] * f + LightAmbient[2];
		f = prod_scal(light_dir,&prail2n[i]);
		if(f<=0.0f) f = 0.0f;
		prail2n[i].x = LightDiffuse[0] * f + LightAmbient[0];
		prail2n[i].y = LightDiffuse[1] * f + 0.0f * LightAmbient[1];
		prail2n[i].z = LightDiffuse[2] * f + LightAmbient[2];
	}
	for(i=0 ; i<nbLine*4 ; i++){
		f = prod_scal(light_dir,&pcyln[i]);
		if(f<=0.0f) f = 0.0f;
		pcyln[i].x = LightDiffuse[0] * f + LightAmbient[0];
		pcyln[i].y = LightDiffuse[1] * f + 0.0f * LightAmbient[1];
		pcyln[i].z = LightDiffuse[2] * f + LightAmbient[2];
	}
	for(i=0 ; i<nbLiens*12 ; i++){
		f = prod_scal(light_dir,&pliensn[i]);
		if(f<=0.0f) f = 0.0f;
		pliensn[i].x = LightDiffuse[0] * f + LightAmbient[0];
		pliensn[i].y = LightDiffuse[1] * f + 0.0f * LightAmbient[1];
		pliensn[i].z = LightDiffuse[2] * f + LightAmbient[2];
	}
}

void 	RollerCoaster::drawTuyau0(point *ptuyau, point *pnorm, int nbLine, int nbCote, float *mat, point *pline){
	int i0,i,j,k;
	point v;
	int flag1,flag2;
	int index;
	mult_vect(&v,mat,&pline[0]);
	flag1 = (v.x<-1.1f || v.x>1.1f || v.y<-1.2f || v.y>1.1f || v.z>0.0f);
	i0 = 0;
	for(i=0 ; i<nbLine ; i++){
		mult_vect(&v,mat,&pline[(i+1)%nbLine]);
		flag2 = (v.x<-1.1f || v.x>1.1f || v.y<-1.2f || v.y>1.1f || v.z>0.0f);
		if(flag1 && flag2){i0 = (i+1)%nbLine; continue;}
		glBegin(GL_QUAD_STRIP);
		for(j=0 ; j<=nbCote ; j++){
			k = j % nbCote;
			index = ((i+1)%nbLine)*nbCote+k;
			glColor3fv((float*)&pnorm[index]);
			glVertex3fv((float*)&ptuyau[index]);
			index = i0*nbCote+k;
			glColor3fv((float*)&pnorm[index]);
			glVertex3fv((float*)&ptuyau[index]);
		}
		glEnd();
		i0 = (i+1)%nbLine;
		flag1 = flag2;
		if((v.z < -2.0f) && i<(nbLine-2) && (i%2)==0) i++;
	}
}

void 	RollerCoaster::drawTuyau(point *ptuyau, point *pnorm, int nbLine, int nbCote, float *mat, point *pline){
	int i0,i,j,k;
	point v;
	int flag1,flag2;
	int shl,shl0;
	int index;
	mult_vect(&v,mat,&pline[0]);
	flag1 = (v.x<-1.1f || v.x>1.1f || v.y<-1.2f || v.y>1.1f || v.z>0.0f);
	i0 = 0;
	shl0 = 0;
	for(i=0 ; i<nbLine ; i++){
		mult_vect(&v,mat,&pline[(i+1)%nbLine]);
		flag2 = (v.x<-1.1f || v.x>1.1f || v.y<-1.2f || v.y>1.1f || v.z>0.0f);
		if((v.z >= -2.0f)||(i==nbLine-1)) shl = 0; else shl = 1;
		if(flag1 && flag2){
			i0 = (i+1)%nbLine;
			shl0 = shl;
			flag1 = flag2;
			continue;
		}
		if(shl && shl0){
			glBegin(GL_QUAD_STRIP);
			for(j=0 ; j<=nbCote ; j+=2){
				k = j % nbCote;
				index = ((i+1)%nbLine)*nbCote+k;
				glColor3fv((float*)&pnorm[index]);
				glVertex3fv((float*)&ptuyau[index]);
				index = i0*nbCote+k;
				glColor3fv((float*)&pnorm[index]);
				glVertex3fv((float*)&ptuyau[index]);
			}
			glEnd();
		}
		else{
			if(shl && !shl0){
				for(j=0 ; j<nbCote ; j+=2){
					glBegin(GL_TRIANGLE_STRIP);
					k = (j+2) % nbCote;
					index = i0*nbCote+k;
					glColor3fv((float*)&pnorm[index]);
					glVertex3fv((float*)&ptuyau[index]);
					index = ((i+1)%nbLine)*nbCote+k;
					glColor3fv((float*)&pnorm[index]);
					glVertex3fv((float*)&ptuyau[index]);
					k = j+1;
					index = i0*nbCote+k;
					glColor3fv((float*)&pnorm[index]);
					glVertex3fv((float*)&ptuyau[index]);
					k = j;
					index = ((i+1)%nbLine)*nbCote+k;
					glColor3fv((float*)&pnorm[index]);
					glVertex3fv((float*)&ptuyau[index]);
					index = i0*nbCote+k;
					glColor3fv((float*)&pnorm[index]);
					glVertex3fv((float*)&ptuyau[index]);
					glEnd();
				}
			}
			else{
				if(!shl && shl0){
					for(j=0 ; j<nbCote ; j+=2){
						glBegin(GL_TRIANGLE_STRIP);
						k = j;
						index = ((i+1)%nbLine)*nbCote+k;
						glColor3fv((float*)&pnorm[index]);
						glVertex3fv((float*)&ptuyau[index]);
						index = i0*nbCote+k;
						glColor3fv((float*)&pnorm[index]);
						glVertex3fv((float*)&ptuyau[index]);
						k = j+1;
						index = ((i+1)%nbLine)*nbCote+k;
						glColor3fv((float*)&pnorm[index]);
						glVertex3fv((float*)&ptuyau[index]);
						k = (j+2) % nbCote;
						index = i0*nbCote+k;
						glColor3fv((float*)&pnorm[index]);
						glVertex3fv((float*)&ptuyau[index]);
						index = ((i+1)%nbLine)*nbCote+k;
						glColor3fv((float*)&pnorm[index]);
						glVertex3fv((float*)&ptuyau[index]);
						glEnd();
					}
				}
				else{
					glBegin(GL_QUAD_STRIP);
					for(j=0 ; j<=nbCote ; j++){
						k = j % nbCote;
						index = ((i+1)%nbLine)*nbCote+k;
						glColor3fv((float*)&pnorm[index]);
						glVertex3fv((float*)&ptuyau[index]);
						index = i0*nbCote+k;
						glColor3fv((float*)&pnorm[index]);
						glVertex3fv((float*)&ptuyau[index]);
					}
					glEnd();
				}
			}
		}
		i0 = (i+1)%nbLine;
		shl0 = shl;
		flag1 = flag2;
		if((v.z < -2.0f) && i<(nbLine-2) && (i%2)==0) i++;
	}
}

void 	RollerCoaster::drawLiens(point *pliens, point *pliensn, int nbLiens, float *mat, point *pnliens){
	int i,j,k;
	point v;
	int flag1,flag2,flag3;
	int loin;
	for(i=0 ; i<nbLiens ; i++){
		mult_vect(&v,mat,&pnliens[i*3]);
		flag1 = (v.x<-1.0f || v.x>1.0f || v.y<-1.0f || v.y>1.0f || v.z>0.0f);
		loin = (v.z < -2.0f);
		mult_vect(&v,mat,&pnliens[i*3+1]);
		flag2 = (v.x<-1.0f || v.x>1.0f || v.y<-1.0f || v.y>1.0f || v.z>0.0f);
		mult_vect(&v,mat,&pnliens[i*3+2]);
		flag3 = (v.x<-1.0f || v.x>1.0f || v.y<-1.0f || v.y>1.0f || v.z>0.0f);
		if(!(flag1&&flag2)){
			glBegin(GL_QUAD_STRIP);
			for(j=0 ; j<5 ; j++){
				if(loin && j==3) j++;
				k = j % 4;
				glColor3fv((float*)&pliensn[i*12+4+k]);
				glVertex3fv((float*)&pliens[i*12+4+k]);
				glColor3fv((float*)&pliensn[i*12+k]);
				glVertex3fv((float*)&pliens[i*12+k]);
			}
			glEnd();
		}
		if(!(flag1&&flag3)){
			glBegin(GL_QUAD_STRIP);
			for(j=0 ; j<5 ; j++){
				if(loin && j==3) j++;
				k = j % 4;
				glColor3fv((float*)&pliensn[i*12+k]);
				glVertex3fv((float*)&pliens[i*12+k]);
				glColor3fv((float*)&pliensn[i*12+8+k]);
				glVertex3fv((float*)&pliens[i*12+8+k]);
			}
			glEnd();
		}
	}
}

void 	RollerCoaster::drawSkyAndGround(int nbCote, float size, float texsize,float bas, float haut){ //nbCote = nombre de cots du cylindre
	int i;
	float x,y;
	float c,s;
	float cc,ss;

	glBegin(GL_QUAD_STRIP);
	glColor3f(0.5f,0.5f,0.7f);
	glVertex3f(size,0.0f,bas);
	glColor3f(0.3f,0.3f,0.7f);
	glVertex3f(size,0.0f,haut);
	for(i=1 ; i<nbCote ; i++){
		x = size * cos((float)i*2.0f*M_PI/(float)nbCote);
		y = size * sin((float)i*2.0f*M_PI/(float)nbCote);
		glColor3f(0.5f,0.5f,0.7f);
		glVertex3f(x,y,bas);
		glColor3f(0.3f,0.3f,0.7f);
		glVertex3f(x,y,haut);
	}
	glColor3f(0.5f,0.5f,0.7f);
	glVertex3f(size,0.0f,bas);
	glColor3f(0.3f,0.3f,0.7f);
	glVertex3f(size,0.0f,haut);
	glEnd();

	/* OOOKKK Draw the ground with many triangles*/
   glEnable(GL_TEXTURE_2D);
   glBmpBind(&grass);
   glColor3f(1.0f,1.0f,1.0f);
   glPushMatrix();
   glTranslatef(1.2f*texsize,1.2f*texsize,0.0f );
   glBegin(GL_POLYGON);
   for(i=0 ; i<nbCote ; i++){
      c   = cos((float)i*2.0f*M_PI/(float)nbCote);
      s   = sin((float)i*2.0f*M_PI/(float)nbCote);
      cc  = texsize*c;
      ss  = texsize*s;
      glTexCoord2f(cc,ss);
      glVertex3f(size*c,size*s,bas);
   }
   glEnd();
   glPopMatrix();
   glDisable(GL_TEXTURE_2D);
}

void 	RollerCoaster::drawEverything() {
        float m[16];
	float modelmat[16], projmat[16];

	glGetFloatv(GL_MODELVIEW_MATRIX,modelmat);
	glGetFloatv(GL_PROJECTION_MATRIX,projmat);
	matrix_mult(m,projmat,modelmat);
	m[2] = modelmat[2];
	m[6] = modelmat[6];
	m[10] = modelmat[10];
	m[14] = modelmat[14];
	drawTuyau0(pcyl,pcyln,nbLine,4,m,pline);
	drawTuyau(prail1,prail1n,nbLine,nbCote,m,pline1);
	drawTuyau(prail2,prail2n,nbLine,nbCote,m,pline2);
	drawLiens(pliens,pliensn,nbLiens,m,pnliens);
	poteau_draw();
	drawSkyAndGround(4,800.0f,40.0f,0.0f,10.0f);
}

void 	RollerCoaster::fade(float opacity) {
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glColor4f(0.0f,0.0f,0.0f,opacity);
	glRectf(-1.0f, -1.0f, 1.0f, 1.0f);
	glDisable(GL_BLEND);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
}

void	RollerCoaster::DrawGLScene0(unsigned long time)
{
	unsigned long	diftime;
	float 			angle,dist;

	angle = (float)(time - firsttime) / 100;			/* angle in degree */
	angle = (angle + 180.0f) / 180.0f * M_PI;			/* angle in radian */

	/* compute camera distance from the origin */
	
	if (maxDist*tan(aperture*M_PI/360.0f)*2.0f < maxZ) dist = maxZ / tan(aperture*M_PI/360.0f) / 2.0f;
	else dist = 0.5f + maxDist;

	glLoadIdentity();
	gluLookAt(dist*cos(angle),dist*sin(angle),maxZ,0.0f,0.0f,maxZ+1.0f-dist*tan(aperture*M_PI/360.0f),0.0f,0.0f,1.0f);
	drawEverything();

	/* fade in at the beginning */

	if (time - firsttime < 100) {
		diftime = time - firsttime;
		fade(1.0f-(float)diftime / 100.0f);
	}

	/* fade out at the end */

	if(time - firsttime > 34000)
	{
		diftime = time - firsttime - 34000;
		fade((float)diftime / 2000.0f);
	}
}

void	RollerCoaster::DrawGLScene(unsigned long current,GLfloat roll_angle, GLfloat pitch_angle)
{
	float t;
	point p;
	point position, eye;
	point ddv,ddv1,ddv2;

	while (scene_abscisse > pnorme[(scene_index+1)%nbLine])
	{
		scene_abscisse -= pnorme[(scene_index+1)%nbLine];
		scene_index++;
		scene_index %= nbLine;
	}

	t = scene_abscisse / pnorme[(scene_index+1)%nbLine];
	position = pline[scene_index];
	sub(&p,&pline[(scene_index+1)%nbLine],&pline[scene_index]);
	add_mult(&position,t,&p);
	ddv1 = ptraj[scene_index];
	sub(&p,&ptraj[(scene_index+1)%nbLine],&ptraj[scene_index]);
	add_mult(&ddv1,t,&p);
	normalize(&ddv1);                /**** ddv1 vector points upward ****/
	ddv = ptang[scene_index];
	sub(&p,&ptang[(scene_index+1)%nbLine],&ptang[scene_index]);
	add_mult(&ddv,t,&p);
	normalize(&ddv);                 /**** ddv vector points forward ****/
	p = ddv1;
	rotate(&ddv1, &ddv, pcourbure[scene_index]+t*(pcourbure[(scene_index+1)%nbLine]-pcourbure[scene_index]), &p);
	prod_vect(&ddv2,&ddv,&ddv1);     /**** ddv2 vector points rightward ****/
	/* compute camera position */
	mult(&eye,0.2f,&ddv1);
	add(&eye,&eye,&position);

	/* rotate camera */
	if (roll_angle!=0.0f) { /* Rotate the front and side vectors around the up vector */
	    p = ddv; /* Front vector */
	    rotate(&ddv, &ddv1, roll_angle, &p);
	    p = ddv2; /* Side vector */
	    rotate(&ddv2, &ddv1, roll_angle, &p);
	} else if (pitch_angle!=0.0f) { /* Rotate the up and front vectors around the side vector. */
	    p = ddv; /* Front vector */
	    rotate (&ddv, &ddv2, pitch_angle, &p);
	    p = ddv1; /* Up vector */
	    rotate (&ddv1, &ddv2, pitch_angle, &p);
	}

	glLoadIdentity();
	gluLookAt(eye.x, eye.y, eye.z, eye.x+ddv.x, eye.y+ddv.y, eye.z+ddv.z, ddv1.x, ddv1.y, ddv1.z);
	drawEverything();

	/* fade in at the beginning of the ride */

	if (current - firsttime < 100) {
		fade(1.0f-(float)(current - firsttime) / 100.0f);
	}
	
	if(scene_flag)
	{
		if(scene_oldz-position.z > 0.0f) scene_flag = 0;
		else scene_vitesse = 0.3f;
	}
	else
	{
		if(scene_index >= brakeSegment || scene_flag_stopping)
		{
			scene_flag_stopping = 1;
			if(scene_vitesse > 0.0f)
			{
				scene_vitesse = scene_vitesse*scene_vitesse - 5.6f * (float)(current - scene_oldtime) / 1000.0f;
				if(scene_vitesse <= 0.0f)
				{
					scene_vitesse = 0.0f;
					scene_stoptime = current;
				}
				else scene_vitesse = sqrt(scene_vitesse);
			}
			else
			{
				if(current - scene_stoptime > 5000)
				{
					scene_flag_stopping = 0;
					scene_flag = 1;
				}
			}
		}
		else
		{
			scene_vitesse = scene_vitesse*scene_vitesse + 3.0f*(scene_oldz-position.z);
			if(scene_vitesse <= 0.0f) scene_vitesse = 0.0f;
			else scene_vitesse = sqrt(scene_vitesse);
		}
	}

	scene_abscisse += scene_vitesse * (float)(current - scene_oldtime) / 1000.0f;
	scene_oldz = position.z;
	scene_oldtime = current;
}

void	RollerCoaster::ChangeRollerParameters(float viewAngle, float eyeDist, float focalLength)
{
	if (aperture != viewAngle)
	{
		aperture = viewAngle;
		ReSizeGLScene(wndWidth,wndHeight);
	}
	
	aperture = viewAngle;
	eye_sep = eyeDist;
	focallength = focalLength;
}

void	RollerCoaster::DrawRoller(unsigned long current)
{
	glClearColor(0.3f, 0.3f, 0.7f, 0.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	
	switch (stage)
	{
		case 0:
			fpstime = firsttime = current;
			stage++;
			
		case 1:
			if (current - firsttime >= 36000) stage++;
			{
				glViewport(0, 0, wndWidth, wndHeight);
				DrawGLScene0(current);
				nbFrames++;
			}
			break;
			
		case 2:
			scene_index = startSegment;
			scene_abscisse = 0.0f;
			scene_oldz = 0.0f;
			scene_vitesse = 0.0f;
			scene_flag = 1;
			scene_flag_stopping = 0;
			scene_oldtime = current - 1;
			firsttime = current;
			stage++;
			
		case 3:
			glViewport(0, 0, wndWidth, wndHeight);
			DrawGLScene(current,0,0);
			nbFrames++;
			break;
	}
	
	if (showFPS) showFrames(current);
}


void	RollerCoaster::showFrames(unsigned long current)
{
	NSString	*string;
	
	if (current - fpstime > 500)
	{
		fps = (float)nbFrames*1000.0f/(float)(current-fpstime);
		fpstime = current;
		nbFrames = 0;
	}

	string = [NSString stringWithFormat:@"Speed: %4.2f FPS",fps];
	[stringtexture draw:string AtPoint:NSMakePoint(16.0,16.0)];
	
	string = [NSString stringWithFormat:@"(%d x %d) \n%s \n%s",wndWidth,wndHeight,glGetString(GL_RENDERER),glGetString(GL_VERSION)];
	[stringtexture draw:string AtPoint:NSMakePoint(16.0,32.0)];
}
